/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: TCPLayer.cpp 13924 2020-12-09 14:27:20Z RMayer $:

  Description:
    Implementation of the physical layer for TCP interfaces

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2020-11-06  Sending eDELETE notification in connector Deinit().
                This is necessary to be compatible with the netX transport toolkit for
                proper connector unload if running as a DLL.
    2020-06-01  Changed TRACE to be printed by OutputDebugString() function
    2015-08-17  CTCPInterface::Send(), changed position of reset event
                to improve send behaviour
    2009-11-27  Review
    2009-07-28  initial version

**************************************************************************************/

/*****************************************************************************/
/*! \file TCPLayer.cpp                                                       */
/*   Implementation of the physical layer for the TCP interfaces             */
/*****************************************************************************/

#include "StdAfx.h"
#include <winsock2.h>
#include <iphlpapi.h>

#include "TCPLayer.h"
#include "netXConnectorErrors.h"
#include "TCPConfig.h"

/* Link socket library */
#pragma comment(lib, "ws2_32.lib")

/* Link Internet Protocol Helper library */
#pragma comment(lib, "iphlpapi.lib")

/*****************************************************************************/
/*! \addtogroup netX_CONNECTOR_TCP netX TCP Connector                        */
/*! \{                                                                       */
/*****************************************************************************/

/* Global layer object */
extern CTCPLayer* g_pcTCPLayer;

/* Global configuration object */
extern CTCPConfig g_cTCPConfig;

///////////////////////////////////////////////////////////////////////////////////////////
/// Debug trace
///////////////////////////////////////////////////////////////////////////////////////////

/* setup degung output (TRACE function) to windows "debugview" */
#define TRACE odprintf
static void __cdecl odprintf(const char *format, ...)
{
  va_list         vaformat;
  char            szVarstring[MAX_PATH];
  char            szBuffer[MAX_PATH];
  SYSTEMTIME      SystemTime;

  GetLocalTime( &SystemTime);

  /* Don't print the date information to log */
  (void)sprintf_s( szBuffer, MAX_PATH,"TCPIPCON: %.2d:%.2d:%.2d.%.3d: ",
                    SystemTime.wHour,
                    SystemTime.wMinute,
                    SystemTime.wSecond,
                    SystemTime.wMilliseconds);

  va_start( vaformat, format);

  DWORD  dwStrlen = (DWORD)strlen(szBuffer);

  (void)vsnprintf_s( szVarstring, MAX_PATH, MAX_PATH - dwStrlen, format, vaformat);
  va_end( vaformat);

  (void)strcat_s(szBuffer, MAX_PATH, szVarstring);

  /* Use Debugprint() for the output */
  szBuffer[strlen(szBuffer)+1] = '\0';
  szBuffer[strlen(szBuffer)] = '\n';
  OutputDebugString(szBuffer);
}
///////////////////////////////////////////////////////////////////////////////////////////
/// Constructor of TCPLayer
///////////////////////////////////////////////////////////////////////////////////////////
CTCPLayer::CTCPLayer(void)
: m_pfnNotifyCallback(NULL)
, m_pvNotifyUser(NULL)
, m_hInterfaceNotifyThread(NULL)
, m_hInterfaceNotifyThreadStop(NULL)
, m_hInterfaceNotifyEvent(NULL)
{
  // Initialize critical section for the interface map
  InitializeCriticalSection(&m_tcsInterfaceLock);
  InitializeCriticalSection(&m_tcsNotifyQueueLock);

  m_cmInterfaceControlMap.clear();
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Destructor of CTCPLayer
///////////////////////////////////////////////////////////////////////////////////////////
CTCPLayer::~CTCPLayer(void)
{
  // Destroy critical section for the interface map
  DeleteCriticalSection(&m_tcsInterfaceLock);
  DeleteCriticalSection(&m_tcsNotifyQueueLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Create a interface with the given name
///  \param  szInterfaceName  Name of the Interface (e.g. TCP0)
///  \return Reference to interface object, NULL if process failed
///////////////////////////////////////////////////////////////////////////////////////////
CTCPInterface* CTCPLayer::CreateInterface(std::string szInterfaceName)
{
  CTCPInterface* pcIntf = NULL;

  LockInterfaceAccess();

  INTERFACE_MAP::iterator iterIntfContr = m_cmInterfaceControlMap.find(szInterfaceName);

  /* Check if interface is in interface control map */
  if (iterIntfContr != m_cmInterfaceControlMap.end())
  {
    /* return pointer to interface object */
    pcIntf = iterIntfContr->second;

  }

  UnlockInterfaceAccess();

  return pcIntf;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Create a interface with the ip address and tcp port. If an interface with the given
/// configuration already exists, the function returns the existing interface object.
/// \param dwIPAddress    IP address of the interface to create (Network byte order)
/// \param usTCPPort      TCP port of the interface to create (Network byte order)
/// \return Reference to interface object, NULL if process failed
///////////////////////////////////////////////////////////////////////////////////////////
CTCPInterface* CTCPLayer::CreateInterface(DWORD dwIPAddress, USHORT usTCPPort)
{
  CTCPInterface* pcIntf = NULL;

  /* Check if there is an interface with the given IP and port in our interface map */
  pcIntf = FindInterface(dwIPAddress, usTCPPort);

  /* There is no interface with the given IP and port, so create new interface */
  if (NULL == pcIntf)
  {
    /* Create name for new interface (e.g. TCP0) */
    std::string szNewInterfaceName = CreateInterfaceName();

    TRACE("%s: IF-Create - interface unknown, create a new one", szNewInterfaceName.c_str());

    /* Create new interface */
    pcIntf = new CTCPInterface( szNewInterfaceName.c_str(),
                                dwIPAddress,
                                usTCPPort,
                                1000,
                                1000,
                                20000,
                                2000);

    LockInterfaceAccess();
    m_cmInterfaceControlMap.insert(make_pair(szNewInterfaceName, pcIntf));
    UnlockInterfaceAccess();

    /* Signal interface is available */
    m_pfnNotifyCallback( szNewInterfaceName.c_str(), eATTACHED, this, m_pvNotifyUser);

    /* Give some time, so notification can reach device handler */
    Sleep(1000);

    TRACE("%s: IF-Create - successfully: 0x%08x", szNewInterfaceName.c_str(), pcIntf);
  }

  return pcIntf;
}

///////////////////////////////////////////////////////////////////////////////////////////
///  Validate the given interface pointer
///   \param  pcInterface  Pointer to interface object
///   \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CTCPLayer::ValidateInterface ( PCONNECTOR_INTERFACE pcInterface)
{
  long lRet = NXCON_DRV_INVALID_POINTER;

  // Lock interface access
  LockInterfaceAccess();

  for (INTERFACE_MAP::iterator iterIntfContr = m_cmInterfaceControlMap.begin(); iterIntfContr != m_cmInterfaceControlMap.end(); ++iterIntfContr)
  {
    if (iterIntfContr->second == pcInterface)
    {
      lRet = NXCON_NO_ERROR;
      break;
    }
  }

  // Unlock interface access
  UnlockInterfaceAccess();

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Locks access to TCP interfaces
///////////////////////////////////////////////////////////////////////////////////////////
void CTCPLayer::LockInterfaceAccess ( void)
{
  EnterCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Unlocks access to TCP interfaces
///////////////////////////////////////////////////////////////////////////////////////////
void CTCPLayer::UnlockInterfaceAccess ( void)
{
  LeaveCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Layer initialization (Initialize Winsock DLL and create TCP interfaces)
///  \param  pfnDevNotifyCallback   Callback for notifications
///  \param  pvUser                 User pointer
///  \return NXCOM_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CTCPLayer::Init( PFN_NETXCON_DEVICE_NOTIFY_CALLBACK pfnDevNotifyCallback, void* pvUser)
{
  /* Check if connector is enabled */
  if (!g_cTCPConfig.IsConnectorEnabled())
  {
    TRACE("Connector Init - failed: lRet: NXCON_DRV_DISABLED");
    return NXCON_DRV_DISABLED;
  }

  if( !pfnDevNotifyCallback)
    return NXCON_DRV_INVALID_PARAMETER;

  if ( m_pfnNotifyCallback)
    return NXCON_DRV_WAS_OPENED_BEFORE;

  WSADATA tWsaData  = {0};            /* WSADATA data structure of the Windows Sockets implementation */
  long    lRet      = NXCON_NO_ERROR;

  if(0 != WSAStartup(MAKEWORD(1,0), &tWsaData))
  {
    TRACE("Connector Init - failed, WSAStartup() returned != 0, lRet: NXCON_DRV_INIT_ERROR");
    lRet = NXCON_DRV_INIT_ERROR;

  } else
  {
    m_pfnNotifyCallback = pfnDevNotifyCallback;
    m_pvNotifyUser      = pvUser;

    /* Create copy from global config object */
    g_cTCPConfig.Lock();
    CTCPConfig cTCPConfig = g_cTCPConfig;
    g_cTCPConfig.Unlock();

    /* Create  Thread and Events */
    m_hInterfaceNotifyThreadStop = CreateEvent(NULL, FALSE, FALSE, NULL);
    m_hInterfaceNotifyEvent      = CreateEvent(NULL, FALSE, FALSE, NULL);

    /* Create Interface Notification Thread to handle interface attachments */
    m_hInterfaceNotifyThread     = CreateThread(NULL, 0, InterfaceNotifyThread, this, 0, NULL);

    /* Get timeout for connection establishment */
    unsigned long ulConnectTimeout = cTCPConfig.GetConnectTimeout();

    /* Enumerate configured interfaces and create interface objects */
    for (CTCPConfig::iterator iterIntf = cTCPConfig.begin(); iterIntf != cTCPConfig.end(); ++iterIntf)
    {
      /* If interface is excluded proceed with next interface */
      if (iterIntf.IsExcluded())
        continue;

      for (DWORD dwIPCurrent =  iterIntf.GetIPRangeBegin(); dwIPCurrent <=  iterIntf.GetIPRangeEnd(); ++dwIPCurrent)
      {
        /* Create name for new interface (e.g. TCP0) */
        std::string szInterfaceName = CreateInterfaceName();

        /* Create new interface */
        CTCPInterface* pcIntf = new CTCPInterface( szInterfaceName.c_str(),
                                                   htonl(dwIPCurrent),
                                                   iterIntf.GetTCPPort(),
                                                   ulConnectTimeout,
                                                   iterIntf.GetSendTimeout(),
                                                   iterIntf.GetResetTimeout(),
                                                   iterIntf.GetKeepAliveTimeout());

        /* Insert interface in control map */
        LockInterfaceAccess();
        m_cmInterfaceControlMap.insert(make_pair(szInterfaceName, pcIntf));
        UnlockInterfaceAccess();

        /* Signal interface is available */
        m_pfnNotifyCallback( szInterfaceName.c_str(), eATTACHED, this, m_pvNotifyUser);

        /* More than NXCON_MAX_INTFACE_CNT are not allowed */
        if (NXCON_MAX_INTFACE_CNT < m_cmInterfaceControlMap.size())
        {
          TRACE("Connector Init - failed, NXCON_MAX_INTFACE_CNT (%d) reached", NXCON_MAX_INTFACE_CNT);
          break;
        }
      }
    }
  }

  /* If something was wrong deinitialize the connector */
  if (NXCON_NO_ERROR != lRet)
  {
    Deinit();
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Create a name for a new interface ( TCP0/TCP1 ...)
/// \return Interface name
///////////////////////////////////////////////////////////////////////////////////////////
std::string CTCPLayer::CreateInterfaceName( void)
{
  std::string szRet;
  char        szName[NXCON_MAX_LENGTH_CONNECTOR_IDENTIFIER+1] = "";
  DWORD       dwInterfaceCnt = 0;

  /* Create unique interface name */
  do
  {
    /* Append index to connector identifier */
    _snprintf(szName, sizeof(szName)/sizeof(*szName), "%s%d", TCP_CONNECTOR_IDENTIFIER, dwInterfaceCnt++);

    /* Check if interface is in interface control map */
  } while ( (dwInterfaceCnt                       <= NXCON_MAX_INTFACE_CNT        ) &&
            (m_cmInterfaceControlMap.find(szName) != m_cmInterfaceControlMap.end())    );

  /* Assign interface name */
  if (dwInterfaceCnt <= NXCON_MAX_INTFACE_CNT)
  {
    szRet.assign(szName);
  } else
  {
    szRet.clear();
  }

  return szRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Layer deinitialization (Deinit and remove TCP interfaces)
///////////////////////////////////////////////////////////////////////////////////////////
void CTCPLayer::Deinit(void)
{
  TRACE("Connector Deinit - Stop communication thread and free all interfaces");

  /* Kill any pending receiver thread */
  if(NULL != m_hInterfaceNotifyThread)
  {
    /* Signal communcation thread to stop */
    SetEvent(m_hInterfaceNotifyThreadStop);

    /* Wait until communcation thread terminates */
    if(WAIT_OBJECT_0 != WaitForSingleObject(m_hInterfaceNotifyThread, CONNECTOR_THREAD_TIMEOUT))
      TerminateThread(m_hInterfaceNotifyThread, MAXDWORD);

    CloseHandle(m_hInterfaceNotifyThreadStop);
    CloseHandle(m_hInterfaceNotifyThread);
    CloseHandle(m_hInterfaceNotifyEvent);

    m_hInterfaceNotifyThreadStop = NULL;
    m_hInterfaceNotifyThread     = NULL;
    m_hInterfaceNotifyEvent      = NULL;
  }

  /* Delete interface objects */
  LockInterfaceAccess();

  for(INTERFACE_MAP::iterator iterIntContr = m_cmInterfaceControlMap.begin(); iterIntContr != m_cmInterfaceControlMap.end(); ++iterIntContr)
  {
    CTCPInterface* pcIntf = iterIntContr->second;

    EnterCriticalSection(&this->m_tcsNotifyQueueLock);
    /* We have to send a eDELETED notification to the netXTransport Layer */
    TRACE("%s: LY-Deinit - send delete Notification: eDELETED!", pcIntf->m_szInterfaceName.c_str());

    // Signal interface deleted
    m_pfnNotifyCallback( pcIntf->m_szInterfaceName.c_str(), eDELETED, this, this->m_pvNotifyUser);
    LeaveCriticalSection(&this->m_tcsNotifyQueueLock);

    iterIntContr->second->Stop();
    delete iterIntContr->second;
    iterIntContr->second = NULL;
  }

  m_cmInterfaceControlMap.clear();
  UnlockInterfaceAccess();

  m_pfnNotifyCallback = NULL;
  m_pvNotifyUser      = NULL;

  WSACleanup();
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Reload configuration of the TCP connector
///////////////////////////////////////////////////////////////////////////////////////////
void CTCPLayer::Reinit(void)
{
  /* Create copy from global config object */
  g_cTCPConfig.Lock();
  CTCPConfig cTCPConfig = g_cTCPConfig;
  g_cTCPConfig.Unlock();

  /* Lock interface control map */
  LockInterfaceAccess();

  /* Check if interfaces are still in configuration */
  for (INTERFACE_MAP::iterator iterInterface = m_cmInterfaceControlMap.begin(); iterInterface != m_cmInterfaceControlMap.end(); ++iterInterface)
  {
    CTCPInterface* pcIntf = iterInterface->second;

    /* Continue with next interface if interface object is invalid */
    if ( INADDR_NONE == pcIntf->GetIPAddress())
      continue;

    /* Mark interface object invalid, if there is no matching configuration entry */
    /* If connector is disabled, all interfaces are marked invalid                */
    if ( (cTCPConfig.end() == cTCPConfig.find(pcIntf->GetIPAddress(), pcIntf->GetTCPPort())) ||
         (!cTCPConfig.IsConnectorEnabled())                                                     )
    {
      /* Set invalid IP address and TCP port */
      pcIntf->SetIPAddress(INADDR_NONE);
      pcIntf->SetTCPPort(0);
      /* Stop interface and signal upper layer that this interface is not more available */
      pcIntf->Stop();
      m_pfnNotifyCallback( pcIntf->m_szInterfaceName.c_str(), eDELETED, this, m_pvNotifyUser);
    }
  }

  /* Enumerate interface configurations */
  for (CTCPConfig::iterator iterConfig = cTCPConfig.begin(); iterConfig != cTCPConfig.end(); ++iterConfig)
  {
    /* Check if connector is enabled */
    if (!cTCPConfig.IsConnectorEnabled())
      break;

    /* If interface is excluded proceed with next interface */
    if (iterConfig.IsExcluded())
      continue;

    for (DWORD dwIPCurrent =  iterConfig.GetIPRangeBegin(); dwIPCurrent <=  iterConfig.GetIPRangeEnd(); ++dwIPCurrent)
    {
      /* Check if there is an interface object with the given configuration */
      if (FindInterface( htonl(dwIPCurrent), iterConfig.GetTCPPort()))
      {
        /* Interface object with the given configuration exists, continue with next configuration */
        continue;

      } else
      {
        /* There is no interface object with the given configuration.
           Reconfigure invalid interface object or create new one     */
        CTCPInterface* pcIntf = FindInterface( INADDR_NONE, 0);

        /* If there is an invalid interface object, reconfigure it.
           Otherwise create a new interface object                  */
        if (pcIntf)
        {
          pcIntf->SetIPAddress(htonl(dwIPCurrent));
          pcIntf->SetTCPPort(iterConfig.GetTCPPort());
          pcIntf->SetSendTimeout(iterConfig.GetSendTimeout());
          pcIntf->SetResetTimeout(iterConfig.GetResetTimeout());
          pcIntf->SetKeepAliveTimeout(iterConfig.GetKeepAliveTimeout());

        } else
        {
          /* Create name for new interface (e.g. TCP0) */
          std::string szInterfaceName = CreateInterfaceName();

          /* Create new interface */
          pcIntf = new CTCPInterface( szInterfaceName.c_str(),
                                      htonl(dwIPCurrent),
                                      iterConfig.GetTCPPort(),
                                      cTCPConfig.GetConnectTimeout(),
                                      iterConfig.GetSendTimeout(),
                                      iterConfig.GetResetTimeout(),
                                      iterConfig.GetKeepAliveTimeout());

          /* Insert interface in control map */
          m_cmInterfaceControlMap.insert(make_pair(szInterfaceName, pcIntf));
        }

        /* Signal interface is available */
        m_pfnNotifyCallback( pcIntf->m_szInterfaceName.c_str(), eATTACHED, this, m_pvNotifyUser);
      }
    }
  }

  UnlockInterfaceAccess();
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Signal attachment or detachment of an interface
///////////////////////////////////////////////////////////////////////////////////////////
void CTCPLayer::SetNotificationEvent( std::string& szInterfaceName, NETX_INTERFACE_NOTIFICATION_E eNotification)
{
  INTERFACE_NOTIFY_T tNotify;
  tNotify.szInterfaceName = szInterfaceName;
  tNotify.eNotification   = eNotification;

  EnterCriticalSection(&m_tcsNotifyQueueLock);
  m_cqNotifyQueue.push(tNotify);
  LeaveCriticalSection(&m_tcsNotifyQueueLock);

  SetEvent(m_hInterfaceNotifyEvent);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Interface Notification Thread to handle interface attachments
///  \param  pvParam            Thread parameters ( CTCPLayer*)
///  \return DWORD  0
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CTCPLayer::InterfaceNotifyThread(void* pvParam)
{
  CTCPLayer* pcLayer        = reinterpret_cast<CTCPLayer*>(pvParam);
  bool       fThreadRunning = true;

  HANDLE ahWaitHandles[] = {pcLayer->m_hInterfaceNotifyEvent, pcLayer->m_hInterfaceNotifyThreadStop};

  while(fThreadRunning)
  {
    DWORD dwWaitRes = WaitForMultipleObjects( sizeof(ahWaitHandles) / sizeof(ahWaitHandles[0]),
                                              ahWaitHandles,
                                              FALSE,
                                              INFINITE);
    switch(dwWaitRes)
    {
      case WAIT_OBJECT_0:      // Receive notification event

        // Check notification callback
        if(pcLayer->m_pfnNotifyCallback == NULL)
          break;

        EnterCriticalSection(&pcLayer->m_tcsNotifyQueueLock);

        // iterate over interface control map
        while(!pcLayer->m_cqNotifyQueue.empty())
        {
            INTERFACE_NOTIFY_T* ptNotify = &pcLayer->m_cqNotifyQueue.front();

            // Signal interface is plugged
            TRACE("%s: LY-Notify - Interface notification received, State: %d", ptNotify->szInterfaceName.c_str(), ptNotify->eNotification);
            pcLayer->m_pfnNotifyCallback( ptNotify->szInterfaceName.c_str(), ptNotify->eNotification, pcLayer, pcLayer->m_pvNotifyUser);

            pcLayer->m_cqNotifyQueue.pop();
        }

        LeaveCriticalSection(&pcLayer->m_tcsNotifyQueueLock);

        break;

      case WAIT_OBJECT_0 + 1:  // Received stop event
        fThreadRunning = false;
        break;

      default:
        _ASSERT(false);
        break;
    }
  }

  return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Find interface with the given interface name
///  \param  szInterface Name of the interface to find
///  \return Pointer reference to the interface
///////////////////////////////////////////////////////////////////////////////////////////
CTCPInterface* CTCPLayer::FindInterface(const char* szInterface)
{
  CTCPInterface* pcRet = NULL;

  EnterCriticalSection(&m_tcsInterfaceLock);
  INTERFACE_MAP::iterator iterInterface = m_cmInterfaceControlMap.find(szInterface);

  if(iterInterface != g_pcTCPLayer->m_cmInterfaceControlMap.end())
  {
    pcRet = iterInterface->second;
  }
  LeaveCriticalSection(&m_tcsInterfaceLock);

  return pcRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Find interface with the given IP address and TCP Port
///  \param  ulIpAddress IP address of the interface to find (Network byte order)
///  \param  usPort      TCP port of the interface to find (Network byte order)
///  \return Pointer reference to the interface
///////////////////////////////////////////////////////////////////////////////////////////
CTCPInterface* CTCPLayer::FindInterface(ULONG ulIpAddress, USHORT usPort)
{
  CTCPInterface* pcRet = NULL;

  EnterCriticalSection(&m_tcsInterfaceLock);
  INTERFACE_MAP::iterator iter = m_cmInterfaceControlMap.begin();

  while(iter != m_cmInterfaceControlMap.end())
  {
    CTCPInterface* pcInterface = iter->second;

    if( (pcInterface->GetIPAddress() == ulIpAddress) &&
        (pcInterface->GetTCPPort()   == usPort) )
    {
      pcRet = pcInterface;
      break;
    }

    iter++;
  }

  LeaveCriticalSection(&m_tcsInterfaceLock);

  return pcRet;

}

///////////////////////////////////////////////////////////////////////////////////////////
///   CTCPInterface
///////////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////////
/// Constructor of TCP interface
///   \param  szIntfName         Name of the interface (e.g. TCP0)
///   \param  ulIPAddr           IP address (Network byte order)
///   \param  usTCPPort          TCP Port number (Network byte order)
///   \param  ulConnectTimeout   Connect timeout in ms
///   \param  ulSendTimeout      Send timeout in ms
///   \param  ulResetTimeout     Reset timeout in ms
///   \param  ulKeepAliveTimeout Keep alive timeout in ms
///////////////////////////////////////////////////////////////////////////////////////////
CTCPInterface::CTCPInterface( const char* szIntfName, unsigned long ulIPAddr, unsigned short usTCPPort, unsigned long ulConnectTimeout, unsigned long ulSendTimeout, unsigned long ulResetTimeout, unsigned long ulKeepAliveTimeout)
: m_hSocket(INVALID_SOCKET)
, m_hCommThread(NULL)
, m_hCommThreadStop(NULL)
, m_szInterfaceName(szIntfName)
, m_hWSAEvent(NULL)
, m_hSendEvent(NULL)
, m_ulState(eINTERFACE_STATE_AVAILABLE)
, m_ulKeepAliveTimeout(ulKeepAliveTimeout)
, m_ulSendTimeout(ulSendTimeout)
, m_ulResetTimeout(ulResetTimeout)
, m_ulConnectTimeout(ulConnectTimeout)
{
  InitializeCriticalSection(&m_tcsInterfaceLock);

  m_tSockAddr.sin_family      = AF_INET;
  m_tSockAddr.sin_addr.s_addr = ulIPAddr;
  m_tSockAddr.sin_port        = usTCPPort;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Destructor
///////////////////////////////////////////////////////////////////////////////////////////
CTCPInterface::~CTCPInterface()
{
  DeleteCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Open connection to TCP server
///  \param  hSocket          Valid socket handle
///  \param  ptSockAddr       Endpoint address to which to connect the socket
///  \param  ulConnectTimeout Connect timeout in ms
///  \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CTCPInterface::OpenConnection( SOCKET hSocket, SOCKADDR_IN* ptSockAddr, unsigned long ulConnectTimeout)
{
  long   lRet       = NXCON_NO_ERROR;
  HANDLE hWSAEvent  = WSA_INVALID_EVENT;

  if (WSA_INVALID_EVENT == (hWSAEvent = WSACreateEvent()))
  {
    /* Error creating WSA event */
    lRet = NXCON_DRV_SOCKETERROR;

  } else if (SOCKET_ERROR == WSAEventSelect( hSocket, hWSAEvent, FD_CONNECT))
  {
    /* Association with the network event FD_CONNECT failed */
    WSACloseEvent(hWSAEvent);
    lRet = NXCON_DRV_SOCKETERROR;

  } else if ( (SOCKET_ERROR   == connect(hSocket, (SOCKADDR*)ptSockAddr,sizeof(SOCKADDR))) &&
              (WSAEWOULDBLOCK != WSAGetLastError()) )
  {
    /* Error connecting to server */
    WSACloseEvent(hWSAEvent);
    lRet = NXCON_DRV_CONNECTION_FAILED;

  } else
  {
    HANDLE ahWaitHandles[] = {hWSAEvent};

    DWORD dwWSARet = WSAWaitForMultipleEvents( sizeof(ahWaitHandles) / sizeof(ahWaitHandles[0]),
                                               ahWaitHandles,
                                               FALSE,
                                               ulConnectTimeout,
                                               FALSE);
    switch(dwWSARet)
    {
      /* Connection event FD_CONNECT */
      case WAIT_OBJECT_0:
        {
          WSANETWORKEVENTS tNetworkEvents;
          WSAEnumNetworkEvents(hSocket, hWSAEvent, &tNetworkEvents);

          /* Connection established */
          if ( FD_CONNECT & tNetworkEvents.lNetworkEvents)
          {
            if (0 != tNetworkEvents.iErrorCode[FD_CONNECT_BIT])
            {
              /* Connection failed */
              TRACE(" Connect failed: %s (Error code %u)\n", inet_ntoa(ptSockAddr->sin_addr), tNetworkEvents.iErrorCode[FD_CONNECT_BIT]);
              lRet = NXCON_DRV_CONNECTION_FAILED;
            } else
            {
              TRACE(" Found: %s \n", inet_ntoa(ptSockAddr->sin_addr));
            }
          }
        }
        break;

      /* Connect timeout */
      case WSA_WAIT_TIMEOUT:
        /* Connection failed */
        lRet = NXCON_DRV_CONNECT_TIMEOUT;
        break;

      default:
        break;

    } // Switch

    WSACloseEvent(hWSAEvent);
  } // Connect

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Query current number of half-open outbound TCP connections
///
/// Background information:
/// Microsoft introduced in Windows XP SP2 a limit of concurrent half-open outbound
/// TCP connections (connection attempts) to slow the spread of virus and malware from
/// system to system. This limit is hardcoded in the Windows system tcpip.sys file.
/// Microsoft's limiter makes it impossible for Windows systems to have more than 10
/// concurrent half-open outbound connections; after 10, new connection attempts are put
/// in a queue and forced to wait. Thus, the speed of connection to other PCs is limited.
/// Therefore, the speed of connection to other computers is actually limited.
///
///  \return Current number of half-open outbound TCP connections
///////////////////////////////////////////////////////////////////////////////////////////
unsigned long CTCPInterface::QueryHalfOpenOutboundTCPConnections( void)
{
  unsigned long ulRet     = 0;
  PMIB_TCPTABLE pTcpTable = {0};
  DWORD dwSize            = sizeof (MIB_TCPTABLE);

  pTcpTable = new MIB_TCPTABLE;

  // Make an initial call to GetTcpTable to
  // get the necessary size into the dwSize variable
  if (ERROR_INSUFFICIENT_BUFFER == GetTcpTable(pTcpTable, &dwSize, TRUE))
  {
      delete pTcpTable;
      pTcpTable = (MIB_TCPTABLE *) new char[dwSize];
  }

  // Make a second call to GetTcpTable to get
  // the actual data we require
  if (NO_ERROR == GetTcpTable(pTcpTable, &dwSize, TRUE))
  {
    for (int iIdx = 0; iIdx < (int) pTcpTable->dwNumEntries; iIdx++)
    {
      if (MIB_TCP_STATE_SYN_SENT == pTcpTable->table[iIdx].dwState)
        ++ulRet;
    }
  }

  delete pTcpTable;

  return ulRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Start interface (Enable receiver)
///   \param  pfnReceiveCallback    Callback for the receiving data
///   \param  pvUser                User pointer
///   \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CTCPInterface::Start(PFN_NETXCON_DEVICE_RECEIVE_CALLBACK pfnReceiveCallback, void* pvUser)
{
  long  lRet = NXCON_NO_ERROR;

  /* If interface is already running return immediately */
  if ( eINTERFACE_STATE_RUNNING   == m_ulState)
  {
    TRACE("%s IF-Start - Interface already started, ulState: eINTERFACE_STATE_RUNNING ", m_szInterfaceName.c_str());
    return NXCON_NO_ERROR;
  }

  /* Previous connection is not available anymore. Stop interface before reconnect. */
  if ( eINTERFACE_STATE_NOT_AVAIL == m_ulState)
  {
    Stop();
  }

  m_pvReceiveUserData   = pvUser;
  m_pfnReceiveCallback  = pfnReceiveCallback;

  if (INVALID_SOCKET == (m_hSocket = CreateSocket()))
  {
    /* Create socket failed */
    lRet = NXCON_DRV_SOCKETERROR;

  } else if (NXCON_NO_ERROR == (lRet = OpenConnection(  m_hSocket, &m_tSockAddr, m_ulConnectTimeout)))
  {
    /* Connection established successfully. Create thread to handle communication */
    if (WSA_INVALID_EVENT == (m_hWSAEvent = WSACreateEvent()))
    {
      /* Create socket event failed */
      lRet = NXCON_DRV_NOT_START;

    } else if (SOCKET_ERROR == WSAEventSelect( m_hSocket, m_hWSAEvent, FD_READ | FD_WRITE | FD_CLOSE  ))
    {
      /* Association with the specified set of FD_XXX network events failed */
      lRet = NXCON_DRV_SOCKETERROR;

    } else if (WSA_INVALID_EVENT == (m_hCommThreadStop = WSACreateEvent()))
    {
      /* Create thread stop event failed */
      lRet = NXCON_DRV_NOT_START;

    } else if (NULL == (m_hSendEvent = CreateEvent(NULL, TRUE, FALSE, NULL)))
    {
      /* Create send event failed */
      lRet = NXCON_DRV_NOT_START;

    } else if (NULL == (m_hCommThread = CreateThread(NULL, 0, CommThread, this, 0, NULL)))
    {
      /* Error creating communication handler thread */
      lRet = NXCON_DRV_NOT_START;

    } else
    {
      /* Interface successfully started. Set state to running */
      m_ulState = eINTERFACE_STATE_RUNNING;
    }
  }

  /* Failed to start interface */
  if (NXCON_NO_ERROR != lRet)
  {
    Stop();
    m_ulState = eINTERFACE_STATE_NOT_AVAIL;

    /* Interface start failed */
    TRACE("%s: IF-Start - failed, ulState: eINTERFACE_STATE_NOT_AVAIL, lRet: 0x%08x", m_szInterfaceName.c_str(), lRet);
  }else
  {
    /* Interface start successfully */
    TRACE("%s: IF-Start - Started, ulState: eINTERFACE_STATE_RUNNING, lRet: 0x%08x", m_szInterfaceName.c_str(), lRet);
  }

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Stop interface
///////////////////////////////////////////////////////////////////////////////////////////
void CTCPInterface::Stop( void)
{
  EnterCriticalSection(&m_tcsInterfaceLock);

  /* Kill any pending receiver thread */
  if(NULL != m_hCommThread)
  {
    /* Signal communcation thread to stop */
    WSASetEvent(m_hCommThreadStop);

    /* Wait until communcation thread terminates */
    if(WAIT_OBJECT_0 != WaitForSingleObject(m_hCommThread, CONNECTOR_THREAD_TIMEOUT))
      TerminateThread(m_hCommThread, MAXDWORD);

    CloseHandle(m_hCommThread);

    m_hCommThread     = NULL;
  }

  if (NULL != m_hCommThreadStop)
  {
    WSACloseEvent(m_hCommThreadStop);
    m_hCommThreadStop = NULL;
  }

  if (NULL != m_hWSAEvent)
  {
    WSACloseEvent(m_hWSAEvent);
    m_hWSAEvent = NULL;
  }

  if (NULL != m_hSendEvent)
  {
    CloseHandle(m_hSendEvent);
    m_hSendEvent = NULL;
  }

  /* Close open TCP Socket */
  if(INVALID_SOCKET != m_hSocket)
  {
    closesocket(m_hSocket);
    m_hSocket = INVALID_SOCKET;
  }

  /* Set interface state to stopped */
  m_ulState = eINTERFACE_STATE_STOPPED;

  TRACE("%s: IF-Stop - ulState: eINTERFACE_STATE_STOPPED", m_szInterfaceName.c_str());

  LeaveCriticalSection(&m_tcsInterfaceLock);
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Send Data to interface
///  \param pabData   Data to send
///  \param ulDataLen Length of data to send
///  \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CTCPInterface::Send(unsigned char* pabData, unsigned long ulDataLen)
{
  long          lRet = NXCON_NO_ERROR;

  /* Check if interface is running */
  if (eINTERFACE_STATE_RUNNING != m_ulState)
  {
    TRACE("%s: Send - error interface not running, State: 0x%08x, lRet: 0x%08x", m_szInterfaceName.c_str(), m_ulState, NXCON_DRV_NOT_START);
    return NXCON_DRV_NOT_START;
  }

  if (SOCKET_ERROR == send (m_hSocket, (char*)pabData, ulDataLen, 0))
  {
    /* Sends are not possible because buffer is full */
    if (WSAEWOULDBLOCK == WSAGetLastError())
    {
      TRACE("%s: Send - error, Send buffer Full", m_szInterfaceName.c_str());

      /* Sends are again possible when an FD_WRITE network event is recorded
         in the CommThreadFunc rountine. This in turn will signal the m_hSendEvent */
      if(WaitForSingleObject(m_hSendEvent, m_ulSendTimeout) != WAIT_OBJECT_0)
      {
        /* Error sending packet */
        lRet = NXCON_DRV_SEND_ERROR;
      } else
      {
        send (m_hSocket, (char*)pabData, ulDataLen, 0);
      }
    } else
    {
      /* Error sending packet */
      lRet = NXCON_DRV_SEND_ERROR;
    }
  }

  ResetEvent(m_hSendEvent);

  if (NXCON_NO_ERROR != lRet)
    TRACE("%s: Send - error,  WSA error code: %u", m_szInterfaceName.c_str(), WSAGetLastError());

  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Get short name of the interface ( TCP0/TCP1 ...)
///  \param ulSize Size of string buffer referenced by szName
///  \param szName String buffer for short interface name
///  \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CTCPInterface::GetShortInterfaceName( unsigned long ulSize, char* szName)
{
  long lRet = NXCON_NO_ERROR;
  if(NULL == szName)
  {
    lRet = NXCON_DRV_INVALID_PARAMETER;
  } else if (m_szInterfaceName.length() >= ulSize)
  {
    lRet = NXCON_DRV_BUFFER_TOO_SHORT;
  } else
  {
    strncpy(szName, m_szInterfaceName.c_str(), ulSize);
  }
  return lRet;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Get long interface name ( e.g. IP192.168.1.2:50111 )
///  \param ulSize  Size of buffer referenced by szName
///  \param szName  Reference for the result string
///  \return NXCON_NO_ERROR on success
///////////////////////////////////////////////////////////////////////////////////////////
long CTCPInterface::GetLongInterfaceName( unsigned long ulSize, char* szName)
{
  long lRet = NXCON_NO_ERROR;

  if(NULL == szName)
    return NXCON_DRV_INVALID_PARAMETER;

  _snprintf(szName, ulSize, "IP%s:%d", inet_ntoa(m_tSockAddr.sin_addr), ntohs(m_tSockAddr.sin_port));

  return lRet;
}


///////////////////////////////////////////////////////////////////////////////////////////
/// Create Socket for the used interface
///   \return Socket handle (INVALID_SOCKET if failed)
///////////////////////////////////////////////////////////////////////////////////////////
SOCKET CTCPInterface::CreateSocket( void)
{
  SOCKET hSocket = INVALID_SOCKET;

  if (INVALID_SOCKET != (hSocket = socket(AF_INET, SOCK_STREAM, 0)))
  {
    int    fNoDelay   = 1;
    linger tLinger    = {1,0};

    if ( (SOCKET_ERROR   == setsockopt(hSocket, IPPROTO_TCP, TCP_NODELAY, (const char*)&fNoDelay, sizeof(fNoDelay))) ||
         (SOCKET_ERROR   == setsockopt(hSocket, SOL_SOCKET, SO_LINGER, (const char*)&tLinger, sizeof(tLinger)))         )
    {
      /* Error setting socket options */
      closesocket(hSocket);
      hSocket = INVALID_SOCKET;
    }
  }

  return hSocket;
}

///////////////////////////////////////////////////////////////////////////////////////////
/// Communication thread wrapper
///  \param pvParam Pointer to object instance
///  \return 0
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CTCPInterface::CommThread(void* pvParam)
{
  CTCPInterface* pcInterface = reinterpret_cast<CTCPInterface*>(pvParam);

  return pcInterface->CommThreadFunc();
}


///////////////////////////////////////////////////////////////////////////////////////////
/// Communication thread (Connection management and receiver)
///  \return 0
///////////////////////////////////////////////////////////////////////////////////////////
DWORD CTCPInterface::CommThreadFunc(void)
{
  HANDLE ahWaitHandles[] = {m_hWSAEvent, m_hCommThreadStop};
  DWORD  dwWSARet        = 0;
  bool   fThreadRunning  = true;

  while(fThreadRunning)
  {
    dwWSARet = WSAWaitForMultipleEvents( sizeof(ahWaitHandles) / sizeof(ahWaitHandles[0]),
                                         ahWaitHandles,
                                         FALSE,
                                         INFINITE,
                                         FALSE);
    switch(dwWSARet)
    {
      /* Connection event (open, closed, receive) */
      case WAIT_OBJECT_0:
        {
          WSANETWORKEVENTS tNetworkEvents;
          WSAEnumNetworkEvents(m_hSocket, m_hWSAEvent, &tNetworkEvents);

          /* Connection closed */
          if ( FD_CLOSE & tNetworkEvents.lNetworkEvents)
          {
            TRACE("%s: CommThreadFunc - Connection closed, State: eINTERFACE_STATE_NOT_AVAIL, WSA error: %u", m_szInterfaceName.c_str(), tNetworkEvents.iErrorCode[FD_CLOSE_BIT]);

            m_ulState = eINTERFACE_STATE_NOT_AVAIL;

            /* Close socket */
            closesocket(m_hSocket);
            m_hSocket = INVALID_SOCKET;

            /* Set device unplugged event  */
            if (g_pcTCPLayer)
              g_pcTCPLayer->SetNotificationEvent(m_szInterfaceName, eDETACHED);

            fThreadRunning = false;

          } else
          {
            /* Data received */
            if ( FD_READ & tNetworkEvents.lNetworkEvents)
            {
              if (0 == tNetworkEvents.iErrorCode[FD_READ_BIT])
              {
                /* We have data to read */
                unsigned long  ulDataLen;
                unsigned char* pabRecvBuffer;

                ioctlsocket(m_hSocket, FIONREAD, &ulDataLen);
                pabRecvBuffer = new unsigned char[ulDataLen];

                if (recv(m_hSocket, (char*)pabRecvBuffer, ulDataLen, 0))
                {
                  // Signal data received
                  if( m_pfnReceiveCallback)
                    m_pfnReceiveCallback(pabRecvBuffer, ulDataLen, m_pvReceiveUserData);
                }
                delete[] pabRecvBuffer;
              } else
              {
                TRACE("%s: CommThreadFunc - error, Socket read, WSA error: %u", m_szInterfaceName.c_str(), tNetworkEvents.iErrorCode[FD_READ_BIT]);
              }
            }

            /* Socket ready for transmission */
            if (FD_WRITE & tNetworkEvents.lNetworkEvents)
            {
              if (0 == tNetworkEvents.iErrorCode[FD_WRITE_BIT])
              {
                SetEvent(m_hSendEvent);
              } else
              {
                TRACE("%s: CommThreadFunc - error, Socket write, WSA error: %u", m_szInterfaceName.c_str(), tNetworkEvents.iErrorCode[FD_WRITE_BIT]);
              }
            }
          }
        }
        break;

      /* Exit communication thread */
      case WAIT_OBJECT_0 + 1:
        fThreadRunning = false;
        break;

      default:
        break;
    }
  }

  return 0;
}

/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
